<?php
$userAgent = $_SERVER['HTTP_USER_AGENT'];
if (preg_match('/Android|iPhone|iPad|iPod|Opera Mini|IEMobile|WPDesktop/i', $userAgent)) {
    header('Location: playlist-ui-mobile.php');
    exit;
}
?>
<?php
require 'config.php';
$tracks = $pdo->query("SELECT id, title, artist, camelot, bpm, duration FROM tracks ORDER BY artist, title")->fetchAll(PDO::FETCH_ASSOC);
$trackCount = $pdo->query("SELECT COUNT(*) FROM tracks WHERE bpm IS NOT NULL AND camelot IS NOT NULL AND bpm != '' AND camelot != ''")->fetchColumn();

?>

<!DOCTYPE html>
<html>
<head>
    <script src="https://sdk.scdn.co/spotify-player.js"></script>

    <meta charset="UTF-8">
    <title>Playlist Builder</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://code.jquery.com/ui/1.13.2/themes/base/jquery-ui.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css">
    <style>
        .badge-diagonal { background-color: #8e44ad !important; color: white; }
        .badge-mood     { background-color: #000 !important; color: white; }
        .badge-jaws     { background-color: #5bc0de !important; color: #000; }
        .bpm-triangle   { font-weight: bold; display: block; }
        .spotify-player { margin-bottom: 1rem; }
        #playlistArea .track-entry { padding: 8px; background: #f1f1f1; border: 1px solid #ccc; margin-bottom: 5px; cursor: move; }
    </style>
</head>
<body class="container py-4">
<h2 class="mb-1">🎧 DJ Playlist Builder</h2>
<span class="badge bg-info text-dark mb-4">🎵 <?= $trackCount ?> tracks with metadata</span>

    <!-- Spotify Player -->
    <div id="spotifyPlayer" class="spotify-player">
        <iframe id="spotifyEmbed" src="" width="100%" height="80" frameborder="0" allowtransparency="true" allow="autoplay; clipboard-write; encrypted-media; fullscreen; picture-in-picture"></iframe>
    </div>

    <div class="row">
        <!-- Left Column: Recommendations -->
        <div class="col-md-7">
            <div class="mb-3">
                <label class="form-label">Select Starting Track:</label>
                <select id="trackSelect" class="form-select">
                    <option value="">-- Choose --</option>
                    <?php foreach ($tracks as $t): ?>
                        <?php
                        $seconds = intval(floatval($t['duration']) * 60);
                        $durationFormatted = gmdate("i:s", $seconds);
                        $camelotColors = [ "1A" => "#00bcd4", "2A" => "#03a9f4", "3A" => "#2196f3", "4A" => "#3f51b5", "5A" => "#673ab7",
                                           "6A" => "#9c27b0", "7A" => "#e91e63", "8A" => "#f44336", "9A" => "#ff5722", "10A" => "#ff9800",
                                           "11A" => "#ffc107", "12A" => "#ffeb3b", "1B" => "#4caf50", "2B" => "#8bc34a", "3B" => "#cddc39",
                                           "4B" => "#ffeb3b", "5B" => "#ffc107", "6B" => "#ff9800", "7B" => "#ff5722", "8B" => "#f44336",
                                           "9B" => "#e91e63", "10B" => "#9c27b0", "11B" => "#673ab7", "12B" => "#3f51b5" ];
                        $color = $camelotColors[$t['camelot']] ?? '#ccc';
                        ?>
                        <option value="<?= $t['id'] ?>" data-camelot="<?= $t['camelot'] ?>" data-bpm="<?= $t['bpm'] ?>" data-color="<?= $color ?>">
                            <?= htmlspecialchars($t['artist']) ?> - <?= htmlspecialchars($t['title']) ?> (<?= $t['bpm'] ?>BPM - <?= $t['camelot'] ?>, <?= $durationFormatted ?>)
                        </option>
                    <?php endforeach; ?>
                </select>
<button id="addStartTrackBtn" class="btn btn-outline-primary btn-sm mt-2">➕ Add Starting Track to Playlist</button>
                
            </div>

            <button id="recommendBtn" class="btn btn-primary mb-3">🎯 Recommend Tracks</button>
<div class="row mb-3">
  <div class="col-md-3">
    <label for="mixFilter" class="form-label">Filter by Mix Type:</label>
    <select id="mixFilter" class="form-select">
      <option value="">All</option>
      <option value="Perfect Mix">Perfect Mix</option>
      <option value="-1 Mix">-1 Mix</option>
      <option value="+1 Mix">+1 Mix</option>
      <option value="Energy Boost">Energy Boost</option>
      <option value="Scale Change">Scale Change</option>
      <option value="Diagonal Mix">Diagonal Mix</option>
      <option value="Jaw's Mix">Jaw's Mix</option>
      <option value="Mood Shifter">Mood Shifter</option>
    </select>
  </div>

  <div class="col-md-3">
    <label for="bpmFilter" class="form-label">Filter by BPM:</label>
    <select id="bpmFilter" class="form-select">
      <option value="">All</option>
      <option value="close">Close Match (±2 BPM)</option>
      <option value="wide">Wide Match (±5 BPM)</option>
    </select>
  </div>
</div>
            <table class="table table-striped" id="resultsTable">
                <thead>
                    <tr>
                        <th>Artist</th>
                        <th>Title</th>
                        <th>BPM</th>
                        <th>Camelot</th>
                        <th>Mix</th>
                        <th>🎧</th>
                        <th>➕</th>
                    </tr>
                </thead>
                <tbody><tr><td colspan="7">Waiting...</td></tr></tbody>
            </table>
        </div>

        <!-- Right Column: Playlist Builder -->
        <div class="col-md-5">
            <h5>Your Playlist</h5>
<table id="playlistArea" class="table table-striped">
  <thead>
    <tr>
      <th>Artist</th>
      <th>Title</th>
      <th>BPM</th>
      <th>Camelot</th>
      <th>Mix</th>
      <th>🎧</th>
      <th>➖</th>
    </tr>
  </thead>
  <tbody></tbody>
</table>

            <button id="savePlaylist" class="btn btn-success mt-2">💾 Save Playlist</button>
            <button id="exportPlaylist" class="btn btn-outline-secondary mt-2">⬇️ Export as M3U</button>
        </div>
    </div>

    <script src="https://code.jquery.com/jquery-3.7.0.min.js"></script>
    <script src="https://code.jquery.com/ui/1.13.2/jquery-ui.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>
    <script>
let playlist = [];

// -- Utility functions --
const mixPriority = {
    'Perfect Mix': 1, '-1 Mix': 2, '+1 Mix': 3,
    'Energy Boost': 4, 'Scale Change': 5, 'Diagonal Mix': 6,
    "Jaw's Mix": 7, 'Mood Shifter': 8, '': 9
};

const getBadgeColor = (type) => {
    switch (type) {
        case 'Perfect Mix':
        case '+1 Mix':
        case '-1 Mix': return 'bg-success';
        case 'Energy Boost': return 'bg-warning';
        case 'Scale Change': return 'bg-primary';
        case "Jaw's Mix": return 'badge-jaws';
        case 'Diagonal Mix': return 'badge-diagonal';
        case 'Mood Shifter': return 'badge-mood';
        default: return 'bg-light';
    }
};

function getMixType(from, to) {
    const numMap = { A: 0, B: 1 };
    if (!from || !to) return '';
    const fromNum = parseInt(from);
    const toNum = parseInt(to);
    const fromMode = from.replace(/[0-9]/g, '');
    const toMode = to.replace(/[0-9]/g, '');
    const fromVal = fromNum + (numMap[fromMode] * 100);
    const toVal = toNum + (numMap[toMode] * 100);
    const diff = toVal - fromVal;

    if (diff === 0) return 'Perfect Mix';
    if (diff === 1 || diff === -11) return '+1 Mix';
    if (diff === -1 || diff === 11) return '-1 Mix';
    if ((fromMode === 'A' && toMode === 'B') || (fromMode === 'B' && toMode === 'A')) {
        return 'Scale Change';
    }
    return '';
}


const getCamelotColor = key => ({
    '1A':'#00bcd4','2A':'#03a9f4','3A':'#2196f3','4A':'#3f51b5','5A':'#673ab7',
    '6A':'#9c27b0','7A':'#e91e63','8A':'#f44336','9A':'#ff5722','10A':'#ff9800',
    '11A':'#ffc107','12A':'#ffeb3b','1B':'#4caf50','2B':'#8bc34a','3B':'#cddc39',
    '4B':'#ffeb3b','5B':'#ffc107','6B':'#ff9800','7B':'#ff5722','8B':'#f44336',
    '9B':'#e91e63','10B':'#9c27b0','11B':'#673ab7','12B':'#3f51b5'
}[key] || '#999');

const getBpmColor = diff => {
    const abs = Math.abs(diff);
    if (abs === 0) return '#00cc00';
    if (abs <= 2) return '#99cc00';
    if (abs <= 4) return '#ffcc00';
    if (abs <= 6) return '#ff6600';
    return '#cc0000';
};

// -- Populate recommendations --
$('#recommendBtn').click(function () {
    const id = $('#trackSelect').val();
    const baseBpm = parseFloat($('#trackSelect option:selected').data('bpm'));
    if (!id) return alert("Please select a track.");

    $('#resultsTable tbody').html('<tr><td colspan="7">Loading...</td></tr>');

    $.getJSON('recommend_playlist.php', {track_id: id}, function (data) {
        data.sort((a, b) => {
            const prioA = mixPriority[a.mix_type] ?? 9;
            const prioB = mixPriority[b.mix_type] ?? 9;
            if (prioA !== prioB) return prioA - prioB;
            const bpmA = Math.abs(parseFloat(a.track.bpm) - baseBpm);
            const bpmB = Math.abs(parseFloat(b.track.bpm) - baseBpm);
            return bpmA - bpmB;
        });

        const rows = data.map(row => {
            const t = row.track;
            const bpmDiff = parseFloat(t.bpm) - baseBpm;
            const bpmTriangle = bpmDiff === 0 ? '✔' : (bpmDiff > 0 ? `▲${Math.abs(bpmDiff)}` : `▼${Math.abs(bpmDiff)}`);
            const bpmColor = getBpmColor(bpmDiff);
            const badge = row.mix_type ? `<span class="badge ${getBadgeColor(row.mix_type)}">${row.mix_type}</span>` : '';
            const camelot = `<span class="badge" style="background-color:${getCamelotColor(t.camelot)};color:#fff">${t.camelot}</span>`;

            return `<tr data-mix="${row.mix_type}" data-bpm="${t.bpm}">
                <td>${t.artist}</td>
                <td>${t.title}</td>
                <td>${t.bpm}<div class="bpm-triangle" style="color:${bpmColor}">${bpmTriangle}</div></td>
                <td>${camelot}</td>
                <td>${badge}</td>
                <td><button class="btn btn-sm btn-outline-success addToPlaylist" data-id="${t.id}" data-title="${t.title}" data-artist="${t.artist}" data-bpm="${t.bpm}" data-camelot="${t.camelot}" data-mix="${row.mix_type}" data-spotify="${t.spotify_id}">➕</button></td>
<td>
  <button class="btn btn-sm btn-outline-dark preview" data-spotify="${t.spotify_id}">▶️</button>
  <button class="btn btn-sm btn-outline-primary download-track" data-title="${t.title}" data-artist="${t.artist}">⬇️</button>
</td>
            </tr>`;
        });

        $('#resultsTable tbody').html(rows.join(''));
    });
});

function applyFilters() {
    const selectedMix = $('#mixFilter').val();
    const selectedBpmFilter = $('#bpmFilter').val();
    const baseBpm = parseFloat($('#trackSelect option:selected').data('bpm')) || 0;

    $('#resultsTable tbody tr').each(function () {
        const mixType = $(this).data('mix');
        const bpm = parseFloat($(this).data('bpm'));
        const bpmDiff = Math.abs(bpm - baseBpm);

        let visible = true;

        if (selectedMix && mixType !== selectedMix) {
            visible = false;
        }

        if (selectedBpmFilter === 'close' && bpmDiff > 2) {
            visible = false;
        } else if (selectedBpmFilter === 'wide' && bpmDiff > 5) {
            visible = false;
        }

        $(this).toggle(visible);
    });
}

applyFilters();

$('#mixFilter, #bpmFilter').on('change', applyFilters);

// -- Add to Playlist --
$(document).on('click', '.addToPlaylist', function () {
    const $btn = $(this);
    const row = $btn.closest('tr');
    const id = $btn.data('id');
    const artist = $btn.data('artist');
    const title = $btn.data('title');
    const bpm = parseFloat($btn.data('bpm'));
    const camelot = $btn.data('camelot');
    const mix = $btn.data('mix') || '';
    if ($('#playlistArea tr[data-id="' + id + '"]').length > 0) {
        alert("Track already added to playlist.");
        return;
    }    
    // Append to playlist table
    const badge = mix ? `<span class="badge ${getBadgeColor(mix)}">${mix}</span>` : '';
    const camelotBadge = `<span class="badge" style="background-color:${getCamelotColor(camelot)};color:#fff">${camelot}</span>`;
    const bpmDisplay = `<div class="bpm-triangle" style="color:${getBpmColor(0)}">✔</div>`; // default if no prev

    $('#playlistArea').append(`
        <tr data-id="${id}" data-bpm="${bpm}" data-camelot="${camelot}" data-title="${title}" data-artist="${artist}">
            <td>${artist}</td>
            <td>${title}</td>
            <td>${bpm}${bpmDisplay}</td>
            <td>${camelotBadge}</td>
            <td>${badge}</td>
            <td><button class="btn btn-sm btn-outline-dark preview" data-spotify="${$btn.data('spotify')}">▶️</button></td>
            <td><button class="btn btn-sm btn-outline-danger remove-from-playlist">−</button></td>
        </tr>
    `);

    // Set last added as selected in Select2
    $("#trackSelect").val(id).trigger('change');
$('#recommendBtn').click();  // Auto-trigger recommendations
    

    updatePlaylistDiffs();  // recalculate diffs
});


// -- Remove from Playlist --
$(document).on('click', '.removeFromPlaylist', function () {
    $(this).closest('tr').remove();
});

// -- Spotify Preview Autoplay --
$(document).on('click', '.preview', function () {
    const uri = $(this).data('spotify');
    $('#spotifyEmbed').attr('src', `https://open.spotify.com/embed/track/${uri}?utm_source=generator&autoplay=1`);
});

// -- Drag-n-Drop --
$(function () {
    $("#playlistArea").sortable({
        placeholder: "ui-state-highlight",
        update: function () {
            updatePlaylistDiffs();  // Recalculate mix types and BPM changes after drag
        }
    }).disableSelection();
});

$(document).on('click', '.download-track', function () {
    const title = $(this).data('title');
    const artist = $(this).data('artist');

    $.post('download.php', { title, artist }, function (res) {
        alert(res.status === 'ok' ? '✅ Download started.' : '❌ Download failed: ' + res.message);
    }, 'json');
});


$(document).on('click', '.download-track', function () {
    const title = $(this).data('title');
    const artist = $(this).data('artist');

    $.post('download.php', { title, artist }, function (res) {
        alert(res.status === 'ok' ? '✅ Download started' : '❌ Failed: ' + res.message);
    }, 'json');
});

// -- Select2 with camelot badge --
$('#trackSelect').select2({
    templateResult: function (data) {
        if (!data.id) return data.text;
        const camelot = $(data.element).data('camelot');
        const color = $(data.element).data('color');
        return $(`<span>${data.text.replace(camelot, `<span class="badge" style="background-color:${color};color:#000">${camelot}</span>`)}</span>`);
    },
    templateSelection: function (data) {
        const camelot = $(data.element).data('camelot');
        const color = $(data.element).data('color');
        return $(`<span>${data.text.replace(camelot, `<span class="badge" style="background-color:${color};color:#000">${camelot}</span>`)}</span>`);
    },
    escapeMarkup: m => m
});

function updatePlaylistDiffs() {
    const rows = $('#playlistArea tr');
    rows.each(function (index) {
        const current = $(this);
        const bpm = parseFloat(current.data('bpm')) || 0;
        const camelot = current.data('camelot');
        const mixCell = current.find('td').eq(4); // mix
        const bpmCell = current.find('td').eq(2); // bpm

        if (index === 0) {
            bpmCell.html(`${bpm}<div class="bpm-triangle" style="color:${getBpmColor(0)}">✔</div>`);
            return;
        }

        const prev = $(rows[index - 1]);
        const prevBpm = parseFloat(prev.data('bpm')) || 0;
        const prevCamelot = prev.data('camelot');

        const bpmDiff = bpm - prevBpm;
        const bpmIcon = bpmDiff === 0 ? '✔' : bpmDiff > 0 ? `▲${Math.abs(bpmDiff)}` : `▼${Math.abs(bpmDiff)}`;
        bpmCell.html(`${bpm}<div class="bpm-triangle" style="color:${getBpmColor(bpmDiff)}">${bpmIcon}</div>`);

        const mixType = getMixType(prevCamelot, camelot);
        const mixBadge = mixType ? `<span class="badge ${getBadgeColor(mixType)}">${mixType}</span>` : '';
        mixCell.html(mixBadge);
    });
}
$('#addSelectedToPlaylist').click(function () {
    const selectedId = $('#trackSelect').val();
    if (!selectedId) return alert("Please select a track first.");

    const option = $('#trackSelect option:selected');
    const artist = option.data('artist');
    const title = option.data('title');
    const bpm = option.data('bpm');
    const camelot = option.data('camelot');
    const spotify = option.data('spotify');

    if (playlist.includes(selectedId)) {
        alert("Track already in playlist.");
        return;
    }

    playlist.push(selectedId);

    const camelotBadge = `<span class="badge" style="background-color:${getCamelotColor(camelot)};color:#fff">${camelot}</span>`;

    const newRow = `<tr data-id="${selectedId}">
        <td>${artist}</td>
        <td>${title}</td>
        <td>${bpm}</td>
        <td>${camelotBadge}</td>
        <td></td>
        <td></td>
        <td></td>
        <td>
            <button class="btn btn-sm btn-outline-danger remove-from-playlist" data-id="${selectedId}">−</button>
        </td>
    </tr>`;

    $('#playlistTable tbody').append(newRow);

    // Update dropdown and trigger recommendation refresh
    $('#trackSelect').val(selectedId).trigger('change');
    $('#recommendBtn').click();
});
window.onSpotifyWebPlaybackSDKReady = () => {
  const token = localStorage.getItem('spotify_access_token');
  const player = new Spotify.Player({
    name: 'Megaload DJ Player',
    getOAuthToken: cb => { cb(token); },
    volume: 0.8
  });

  player.connect();
  // Optional: add listeners for state change, errors, etc.
};

$('#addStartTrackBtn').click(function () {
    const selected = $('#trackSelect option:selected');
    const id = selected.val();

    if (!id) {
        alert("Please select a track.");
        return;
    }

    if (playlist.includes(id)) {
        alert("Track already in playlist.");
        return;
    }

    // Build row from select option attributes
    const artist = selected.data('artist') || selected.text().split('-')[0].trim();
    const title = selected.data('title') || selected.text().split('-')[1].split('(')[0].trim();
    const bpm = parseFloat(selected.data('bpm'));
    const camelot = selected.data('camelot');
    const color = selected.data('color');

    const camelotBadge = `<span class="badge" style="background-color:${color};color:#fff">${camelot}</span>`;
    const bpmBadge = `<span class="bpm-triangle" style="color:#00cc00;">✔</span>`; // starting track has no previous
    const mixType = ''; // no mix type for first track

    const row = `<tr data-id="${id}">
        <td>${artist}</td>
        <td>${title}</td>
        <td>${bpm}<br>${bpmBadge}</td>
        <td>${camelotBadge}</td>
        <td>${mixType}</td>
        <td><button class="btn btn-sm btn-outline-dark preview" data-spotify="${selected.data('spotify')}">▶️</button></td>
        <td><button class="btn btn-sm btn-outline-danger remove-from-playlist" data-id="${id}">−</button></td>
    </tr>`;

    $('#playlistArea tbody').append(row);
    playlist.push(id);

    // Update the Select2 dropdown to new track
    $('#trackSelect').val(id).trigger('change');

    // Trigger recommendation refresh
    $('#recommendBtn').click();
});


</script>

</body>
</html>
