<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>🎧 Mix Playlist Builder</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <style>
    body { overflow-x: hidden; }
    .mobile-track {
      background-color: #2c2f33;
      border-left: 5px solid #007bff;
      padding: 1rem;
      margin-bottom: 1rem;
      border-radius: 5px;
      color: white;
    }
    .mix-transition {
      text-align: center;
      color: #0dcaf0;
      font-size: 0.85rem;
      margin: 0.5rem 0;
    }
  #suggestionBox h5 {
    margin-bottom: 1rem;
  }
    
  </style>
</head>
<body class="bg-dark text-white">
<div class="container-fluid">
  <div class="row">
    <div id="sidebar" class="col-md-3 p-3 bg-secondary">
      <h5 class="text-white">🎵 Track Selector</h5>
      <button id="toggleView" class="btn btn-sm btn-warning w-100 mb-3">Toggle Mobile/Desktop View</button>
      <div class="mb-3">
        <select id="startingTrack" class="form-select">
          <option value="">Select starting track</option>
        </select>
        <button id="startMix" class="btn btn-sm btn-primary mt-2 w-100">➕ Add to Mix</button>
      </div>
      <input type="text" id="filterCamelot" class="form-control form-control-sm mb-2" placeholder="Filter by Camelot">
      <input type="text" id="filterBpm" class="form-control form-control-sm mb-2" placeholder="Filter by BPM">
      <input type="text" id="mixName" class="form-control form-control-sm mt-2" placeholder="Enter mix name">
      <button id="saveToDb" class="btn btn-sm btn-outline-success w-100 mt-2">🗄️ Save to Database</button>
      <button id="saveSession" class="btn btn-sm btn-success w-100 mt-2">💾 Save Session</button>
      <button id="loadSession" class="btn btn-sm btn-info w-100 mt-2">📂 Load Session</button>
      <button id="exportCsv" class="btn btn-sm btn-light w-100 mt-2">⬇ Export Mix (CSV)</button>
    </div>
    <div class="col-md-9">
      <div class="d-flex justify-content-between align-items-center p-3">
        <h4 class="mb-0">🎶 Mix Builder</h4>
        <div>⏱ Total Duration: <strong id="mixDuration">00:00</strong></div>
      </div>
      <div id="mainContent"></div>
<div class="mt-4" id="suggestionBox">
  <h5 class="text-info">🎯 Suggested Next Tracks</h5>
  <div id="suggestionContent"></div>
</div>
      
    </div>
  </div>
</div>

<script>
let useMobileView = false;
let totalSeconds = 0;
let lastTrack = null;
let playlist = [];

document.getElementById('toggleView').onclick = () => {
  useMobileView = !useMobileView;
  document.getElementById('mainContent').innerHTML = '';
  if (lastTrack) loadSuggestions(lastTrack);
};

document.getElementById('startMix').onclick = () => {
  const selected = document.getElementById('startingTrack').value;
  if (!selected) return alert('Select a track');
  const track = JSON.parse(selected);
  addToMix(track);
};

function addToMix(track, mixType = '') {
  playlist.push({ ...track, mixType });
  const [min, sec] = track.duration.split(':').map(Number);
  totalSeconds += min * 60 + sec;
  updateTotalDuration();

  if (lastTrack && mixType) {
    const label = document.createElement('div');
    label.className = 'mix-transition';
    label.textContent = `→ ${mixType} (${lastTrack.camelot} → ${track.camelot})`;
    document.getElementById('mainContent').appendChild(label);
  }

  const div = document.createElement('div');
  div.className = 'mobile-track';
  div.innerHTML = `
    <strong>${track.title}</strong><br>
    <small>${track.artist}</small><br>
    <div>${track.camelot} | ${track.bpm || 'N/A'} BPM</div>
    <div>${track.duration}</div>
    ${track.spotify_id ? `<a class="btn btn-sm btn-outline-light mt-2" href="https://open.spotify.com/track/${track.spotify_id}" target="_blank">▶ Preview</a>` : ''}
  `;
  document.getElementById('mainContent').appendChild(div);
  lastTrack = track;
  loadSuggestions(track);
}

function updateTotalDuration() {
  const min = Math.floor(totalSeconds / 60);
  const sec = totalSeconds % 60;
  document.getElementById('mixDuration').textContent = `${min.toString().padStart(2, '0')}:${sec.toString().padStart(2, '0')}`;
}

function loadSuggestions(track) {
  fetch(`get_compatible_tracks.php?camelot=${track.camelot}&bpm=${track.bpm}`)
    .then(res => res.json())
    .then(data => {
      const box = document.getElementById('suggestionContent');
      box.innerHTML = '';

      if (Object.keys(data).length === 0) {
        box.innerHTML = '<em class="text-muted">No suggestions found.</em>';
        return;
      }

      for (const [mixType, tracks] of Object.entries(data)) {
        // Sort by closest BPM
        tracks.sort((a, b) => {
          const diffA = Math.abs((a.bpm || 0) - (track.bpm || 0));
          const diffB = Math.abs((b.bpm || 0) - (track.bpm || 0));
          return diffA - diffB;
        });

        const title = document.createElement('h6');
        title.className = 'text-light mt-3';
        title.innerText = mixType;

        const list = document.createElement('div');

        tracks.forEach((t, index) => {
          const bpmDiff = Math.abs((t.bpm || 0) - (track.bpm || 0));
          const btn = document.createElement('button');
          btn.className = `btn btn-sm w-100 text-start mb-1 ${
            index === 0 ? 'btn-success' : 'btn-outline-light'
          }`;

          btn.innerHTML = `
            ${t.title} — ${t.artist}<br>
            <small>[${t.camelot}] ${t.bpm} BPM | ${t.duration}</small>
            <span class="text-muted float-end">Δ ${bpmDiff} BPM</span>
          `;
          btn.onclick = () => addToMix(t, mixType);
          list.appendChild(btn);
        });

        box.appendChild(title);
        box.appendChild(list);
      }
    });
}



document.getElementById('exportCsv').onclick = () => {
  const rows = [['Mix Type', 'Title', 'Artist', 'Camelot', 'BPM', 'Duration']];
  playlist.forEach(t => {
    rows.push([t.mixType || '', t.title, t.artist, t.camelot, t.bpm, t.duration]);
  });
  const csv = rows.map(r => r.join(',')).join('\\n');
  const blob = new Blob([csv], { type: 'text/csv' });
  const a = document.createElement('a');
  a.href = URL.createObjectURL(blob);
  a.download = 'mix_playlist.csv';
  a.click();
};

document.getElementById('saveSession').onclick = () => {
  localStorage.setItem('mixPlaylist', JSON.stringify({ playlist, totalSeconds }));
  alert('Session saved locally.');
};

document.getElementById('loadSession').onclick = () => {
  const saved = JSON.parse(localStorage.getItem('mixPlaylist') || '{}');
  if (!saved.playlist) return alert('No session found.');
  playlist = saved.playlist;
  totalSeconds = saved.totalSeconds || 0;
  document.getElementById('mainContent').innerHTML = '';
  playlist.forEach((track, i) => {
    if (i > 0) {
      const prev = playlist[i - 1];
      const label = track.mixType || 'Mix';
      const transition = document.createElement('div');
      transition.className = 'mix-transition';
      transition.textContent = `→ ${label} (${prev.camelot} → ${track.camelot})`;
      document.getElementById('mainContent').appendChild(transition);
    }
    const div = document.createElement('div');
    div.className = 'mobile-track';
    div.innerHTML = `
      <strong>${track.title}</strong><br>
      <small>${track.artist}</small><br>
      <div>${track.camelot} | ${track.bpm || 'N/A'} BPM</div>
      <div>${track.duration}</div>
      ${track.spotify_id ? `<a class="btn btn-sm btn-outline-light mt-2" href="https://open.spotify.com/track/${track.spotify_id}" target="_blank">▶ Preview</a>` : ''}
    `;
    document.getElementById('mainContent').appendChild(div);
  });
  updateTotalDuration();
  lastTrack = playlist.at(-1);
  loadSuggestions(lastTrack);
};

document.getElementById('saveToDb').onclick = () => {
  const name = document.getElementById('mixName').value.trim();
  if (!name) return alert('Enter mix name first');
  fetch('save_playlist.php', {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({ name, playlist })
  })
  .then(res => res.json())
  .then(data => {
    if (data.success) alert('✅ Playlist saved!');
    else alert('❌ Error: ' + (data.error || 'Unknown error'));
  });
};

fetch('get_tracks.php')
  .then(res => res.json())
  .then(tracks => {
    const select = document.getElementById('startingTrack');
    tracks.forEach(t => {
      const opt = document.createElement('option');
      opt.value = JSON.stringify(t);
      opt.textContent = `${t.title} — ${t.artist} (${t.duration}) [${t.camelot}] ${t.bpm || 'N/A'} BPM`;
      select.appendChild(opt);
    });
  });
</script>
</body>
</html>