"use strict";

const ipRegex = require('ip-regex');
const tlds = require('tlds');
const ipv4 = ipRegex.v4().source;
const ipv6 = ipRegex.v6().source;
const host = '(?:(?:[a-z\\u00a1-\\uffff0-9][-_]*)*[a-z\\u00a1-\\uffff0-9]+)';
const domain = '(?:\\.(?:[a-z\\u00a1-\\uffff0-9]-*)*[a-z\\u00a1-\\uffff0-9]+)*';
const strictTld = '(?:[a-z\\u00a1-\\uffff]{2,})';
const defaultTlds = `(?:${tlds.sort((a, b) => b.length - a.length).join('|')})`;
const port = '(?::\\d{2,5})?';
let RE2;
let hasRE2;
module.exports = options => {
  options = {
    //
    // attempt to use re2, if set to false will use RegExp
    // (we did this approach because we don't want to load in-memory re2 if users don't want it)
    // <https://github.com/spamscanner/url-regex-safe/issues/28>
    //
    re2: true,
    exact: false,
    strict: false,
    auth: false,
    localhost: true,
    parens: false,
    apostrophes: false,
    trailingPeriod: false,
    ipv4: true,
    ipv6: true,
    returnString: false,
    ...options
  };

  /* istanbul ignore next */
  const SafeRegExp = options.re2 && hasRE2 !== false ? (() => {
    if (typeof RE2 === 'function') return RE2;
    try {
      RE2 = require('re2');
      return typeof RE2 === 'function' ? RE2 : RegExp;
    } catch {
      hasRE2 = false;
      return RegExp;
    }
  })() : RegExp;
  const protocol = `(?:(?:[a-z]+:)?//)${options.strict ? '' : '?'}`;

  // Add option to disable matching urls with HTTP Basic Authentication
  // <https://github.com/kevva/url-regex/pull/63>
  const auth = options.auth ? '(?:\\S+(?::\\S*)?@)?' : '';

  // Add ability to pass custom list of tlds
  // <https://github.com/kevva/url-regex/pull/66>
  const tld = `(?:\\.${options.strict ? strictTld : options.tlds ? `(?:${options.tlds.sort((a, b) => b.length - a.length).join('|')})` : defaultTlds})${options.trailingPeriod ? '\\.?' : ''}`;
  let disallowedChars = '\\s"';
  if (!options.parens) {
    // Not accept closing parenthesis
    // <https://github.com/kevva/url-regex/pull/35>
    disallowedChars += '\\)';
  }
  if (!options.apostrophes) {
    // Don't allow apostrophes
    // <https://github.com/kevva/url-regex/pull/55>
    disallowedChars += "'";
  }
  const path = options.trailingPeriod ? `(?:[/?#][^${disallowedChars}]*)?` : `(?:(?:[/?#][^${disallowedChars}]*[^${disallowedChars}.?!])|[/])?`;

  // Added IPv6 support
  // <https://github.com/kevva/url-regex/issues/60>
  let regex = `(?:${protocol}|www\\.)${auth}(?:`;
  if (options.localhost) regex += 'localhost|';
  if (options.ipv4) regex += `${ipv4}|`;
  if (options.ipv6) regex += `${ipv6}|`;
  regex += `${host}${domain}${tld})${port}${path}`;

  // Add option to return the regex string instead of a RegExp
  if (options.returnString) return regex;
  return options.exact ? new SafeRegExp(`(?:^${regex}$)`, 'i') : new SafeRegExp(regex, 'ig');
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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