const puppeteer = require('puppeteer-extra');
const StealthPlugin = require('puppeteer-extra-plugin-stealth');
const fs = require('fs');
const path = require('path');

puppeteer.use(StealthPlugin());

const COOKIE_FILE = path.join(__dirname, 'cookies.json');
const CHROME_PATH = '/usr/bin/google-chrome';
const SEARCH_URL = 'https://tunebat.com/Search?q=bread+odd+mob+remix';

const USER_AGENTS = [
  'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/122.0.0.0 Safari/537.36',
  'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/15.1 Safari/605.1.15',
  'Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/122.0.0.0 Safari/537.36'
];

const wait = (ms) => new Promise(resolve => setTimeout(resolve, ms));

const loadCookies = async (page) => {
  if (fs.existsSync(COOKIE_FILE)) {
    try {
      const cookies = JSON.parse(fs.readFileSync(COOKIE_FILE, 'utf8'));
      if (!Array.isArray(cookies)) throw new Error('Invalid cookie format');
      await page.setCookie(...cookies);
      console.log('🍪 Cookies loaded');
    } catch (e) {
      console.warn('⚠️ Could not load cookies:', e.message);
    }
  } else {
    console.log('⚠️ No cookies.json found, starting fresh');
  }
};

const saveCookies = async (page) => {
  const cookies = await page.cookies();
  fs.writeFileSync(COOKIE_FILE, JSON.stringify(cookies, null, 2));
  console.log('✅ Cookies saved');
};

const mimicHuman = async (page) => {
  await page.mouse.move(100, 100, { steps: 10 });
  await wait(500);
  await page.mouse.move(200, 300, { steps: 15 });
  await wait(300);
  await page.mouse.move(400, 250, { steps: 25 });

  await page.keyboard.press('ArrowDown');
  await wait(300);
  await page.keyboard.press('ArrowUp');
};

const waitForCloudflare = async (page, context) => {
  console.log(`⏳ [${context}] Waiting for Cloudflare check (20s max)...`);
  let cloudflarePassed = false;

  try {
    await page.waitForFunction(() => {
      return !document.body.innerText.includes('Verifying you are human');
    }, { timeout: 20000 });
    cloudflarePassed = true;
  } catch {
    console.warn(`⚠️ [${context}] Cloudflare check did not complete in 20s`);
  }

  if (!cloudflarePassed) {
    const refreshLink = await page.$('.refresh_link');
    if (refreshLink) {
      console.log(`🔁 [${context}] Clicking ".refresh_link" to reload...`);
      await refreshLink.click();
      await wait(10000); // Wait for refresh
    } else {
      console.warn(`❌ [${context}] No .refresh_link element found — exiting.`);
      return false;
    }
  }

  return true;
};

const runScraper = async () => {
  const browser = await puppeteer.launch({
    headless: false,
    executablePath: CHROME_PATH,
    userDataDir: './my-chrome-profile',
    args: [
      '--no-sandbox',
      '--disable-setuid-sandbox',
      '--disable-blink-features=AutomationControlled',
      '--disable-dev-shm-usage',
      '--disable-infobars',
      '--window-size=1280,800'
    ]
  });

  const page = await browser.newPage();

  const userAgent = USER_AGENTS[Math.floor(Math.random() * USER_AGENTS.length)];
  await page.setUserAgent(userAgent);
  await page.setViewport({ width: 1280 + Math.floor(Math.random() * 100), height: 800 });

  await page.evaluateOnNewDocument(() => {
    Object.defineProperty(navigator, 'webdriver', { get: () => false });
  });

  await loadCookies(page);

  console.log(`🌍 Navigating to: ${SEARCH_URL}`);
  await page.goto(SEARCH_URL, { waitUntil: 'domcontentloaded', timeout: 0 });

  console.log('🧠 Simulating human-like interaction...');
  await mimicHuman(page);

  const cfOkSearch = await waitForCloudflare(page, 'Search page');
  if (!cfOkSearch) {
    await browser.close();
    return;
  }

  try {
    await page.waitForSelector('.ant-row.pDoqI a', { timeout: 15000 });
    console.log('🔍 Search results loaded');
  } catch {
    console.warn('❌ Search results not detected — possibly blocked');
    await browser.close();
    return;
  }

  await saveCookies(page);

  const relativeLink = await page.$eval('.ant-row.pDoqI a', a => a.getAttribute('href'));
  const trackUrl = 'https://tunebat.com' + relativeLink;

  console.log('🎵 Navigating to track info:', trackUrl);
  await page.goto(trackUrl, { waitUntil: 'networkidle2', timeout: 0 });

  await mimicHuman(page);

  const cfOkTrack = await waitForCloudflare(page, 'Track page');
  if (!cfOkTrack) {
    await browser.close();
    return;
  }

  await wait(5000);
  await page.screenshot({ path: 'debug-track-page.png', fullPage: true });

  try {
    await page.waitForSelector('h1.BSDxW', { timeout: 20000 });
    console.log('✅ Track page loaded');
  } catch (err) {
    console.error('❌ Failed to detect track title. Check debug-track-page.png manually.');
    await browser.close();
    return;
  }

  const data = await page.evaluate(() => {
    const getValueByLabel = (label) => {
      const elements = Array.from(document.querySelectorAll('.yIPfN'));
      for (const el of elements) {
        const labelEl = el.querySelector('.ant-typography-secondary');
        if (labelEl && labelEl.innerText.toLowerCase() === label.toLowerCase()) {
          return el.querySelector('.ant-typography')?.innerText.trim();
        }
      }
      return '';
    };

    const getMetricByLabel = (label) => {
      const blocks = Array.from(document.querySelectorAll('._1MCwQ'));
      for (const block of blocks) {
        const metricLabel = block.querySelector('.fd89q')?.innerText?.trim().toLowerCase();
        if (metricLabel === label.toLowerCase()) {
          return block.querySelector('.ant-progress-text')?.innerText.trim();
        }
      }
      return '';
    };

    return {
      title: document.querySelector('h1.BSDxW')?.innerText.trim() || '',
      artist: document.querySelector('h3._6IzUR')?.innerText.trim() || '',
      key: getValueByLabel('key'),
      camelot: getValueByLabel('camelot'),
      bpm: getValueByLabel('bpm'),
      duration: getValueByLabel('duration'),
      popularity: getMetricByLabel('popularity'),
      energy: getMetricByLabel('energy'),
      danceability: getMetricByLabel('danceability'),
      happiness: getMetricByLabel('happiness'),
      acousticness: getMetricByLabel('acousticness'),
      instrumentalness: getMetricByLabel('instrumentalness'),
      liveness: getMetricByLabel('liveness'),
      speechiness: getMetricByLabel('speechiness'),
      loudness: getMetricByLabel('loudness'),
    };
  });

  console.log('🎧 Track Info:', data);

  await saveCookies(page);
  await browser.close();
};

runScraper();
